/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
abdomen_score();

/**
 * Performs the abdomen score calculation for the specified models
 */
function abdomen_score() {
    let template = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = "abdomen";

    let occupants = ["DRIVER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* ABDOMEN_HI_SCORE copies chest */
            let ABDOMEN_HI_SCORE = get_variable_value(status, `CHEST_HI_SCORE`, "int");

            let ABDOMEN_COMPRESSION_HI_LIMIT = 47; // The abdominal compression limit in mm
            let ABDOMEN_COMPRESSION_LO_LIMIT = 65; // The abdominal compression limit in mm
            let ABDOMEN_COMPRESSION_CAPPING_LIMIT = 65; // Capping limit is 65mm (same as lower limit)
            let ABDOMEN_COMPRESSION_GOOD = ABDOMEN_COMPRESSION_HI_LIMIT; //higher performance limit
            let ABDOMEN_COMPRESSION_WEAK = ABDOMEN_COMPRESSION_LO_LIMIT; //lower performance limit

            let max_upp_rib_com = get_variable_value(status, `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_01_VALUE`, "float");
            let max_bottom_rib_com = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_02_VALUE`,
                "float"
            );

            if (max_upp_rib_com == "Missing" || max_upp_rib_com == null) {
                var topc = 0;
            } else {
                var topc = sliding_scale(
                    max_upp_rib_com,
                    ABDOMEN_COMPRESSION_GOOD,
                    ABDOMEN_COMPRESSION_WEAK,
                    ABDOMEN_HI_SCORE,
                    0
                );
            }

            if (max_bottom_rib_com == "Missing" || max_bottom_rib_com == null) {
                var botc = 0;
            } else {
                var botc = sliding_scale(
                    max_bottom_rib_com,
                    ABDOMEN_COMPRESSION_GOOD,
                    ABDOMEN_COMPRESSION_WEAK,
                    ABDOMEN_HI_SCORE,
                    0
                );
            }

            new Variable(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_01_SCORE`,
                `Result from Automotive Assessments Workflow`,
                topc.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_02_SCORE`,
                `Result from Automotive Assessments Workflow`,
                botc.toFixed(3),
                "General",
                false,
                true
            );

            let maxc = Math.max(max_upp_rib_com, max_bottom_rib_com);
            new Variable(
                template,
                `ABDOMEN_MAX_COMPRESSION`,
                `Result from Automotive Assessments Workflow`,
                maxc.toFixed(3),
                "General",
                false,
                true
            );

            let abdomen_score = Math.min(topc, botc);
            new Variable(
                template,
                `ABDOMEN_MAX_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                abdomen_score.toFixed(3),
                "General",
                false,
                true
            );

            let chest_score = get_variable_value(status, `CHEST_MAX_COMPRESSION_SCORE`, "float");
            let chest_abdomen_score = Math.min(chest_score, abdomen_score);
            new Variable(
                template,
                `CHEST_ABDOMEN_FINAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                chest_abdomen_score.toFixed(3),
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_ABDOMEN_CAPPING_LIMIT`,
                `Result from Automotive Assessments Workflow`,
                maxc > ABDOMEN_COMPRESSION_CAPPING_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
